/*
 * Copyright (C) 2017 foo86
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/channel_layout.h"
#include "libavutil/float_dsp.h"
#include "libavutil/thread.h"
#include "libavutil/mem.h"
#include "libavutil/mem_internal.h"
#include "libavutil/opt.h"

#include "codec_internal.h"
#include "internal.h"
#include "get_bits.h"
#include "dolby_e.h"
#include "kbdwin.h"
#include "fft.h"

#define MAX_SEGMENTS    2

#define MAX_GROUPS      8
#define MAX_EXPONENTS   304
#define MAX_MANTISSAS   1024

#define MAX_MSTR_EXP    2
#define MAX_BIAS_EXP    50

enum DBEOutputChannelOrder {
    CHANNEL_ORDER_DEFAULT,
    CHANNEL_ORDER_CODED,
};

typedef struct DBEGroup {
    uint8_t         nb_exponent;
    uint8_t         nb_bias_exp[MAX_MSTR_EXP];
    uint16_t        exp_ofs;
    uint16_t        mnt_ofs;
    const uint8_t   *nb_mantissa;
    uint8_t         imdct_idx;
    uint8_t         imdct_phs;
    uint16_t        win_len;
    uint16_t        dst_ofs;
    uint16_t        win_ofs;
    uint16_t        src_ofs;
} DBEGroup;

typedef struct DBEChannel {
    int     gr_code;
    int     bw_code;

    int         nb_groups;
    int         nb_mstr_exp;
    DBEGroup    groups[MAX_GROUPS];

    int     exp_strategy[MAX_GROUPS];
    int     exponents[MAX_EXPONENTS];
    int     bap[MAX_EXPONENTS];
    int     idx[MAX_EXPONENTS];

    DECLARE_ALIGNED(32, float, mantissas)[MAX_MANTISSAS];
} DBEChannel;

typedef struct DBEDecodeContext {
    const AVClass   *class;
    AVCodecContext  *avctx;
    DBEContext  dectx;

    DBEChannel  channels[MAX_SEGMENTS][MAX_CHANNELS];

    DECLARE_ALIGNED(32, float, history)[MAX_CHANNELS][256];

    FFTContext          imdct[3];
    AVFloatDSPContext   *fdsp;
} DBEDecodeContext;

static const int8_t lfe_channel_tab[MAX_PROG_CONF + 1] = {
     5,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, 4,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  5, 5
};

static const uint8_t ch_reorder_4[4] = { 0, 2, 1, 3 };
static const uint8_t ch_reorder_6[6] = { 0, 2, 4, 1, 3, 5 };
static const uint8_t ch_reorder_8[8] = { 0, 2, 6, 4, 1, 3, 7, 5 };
static const uint8_t ch_reorder_n[8] = { 0, 2, 4, 6, 1, 3, 5, 7 };


static const uint8_t nb_groups_tab[4] = { 1, 8, 7, 1 };

static const uint8_t nb_mstr_exp_tab[4] = { 2, 2, 2, 1 };

static const uint8_t nb_mantissa_38[38] = {
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     2,  2,  2,  2,  2,  2,  3,  3,  3,  4,  4,  4,  5,  5,  6,  6,
     7,  8,  9, 10, 11, 12,
};

static const uint8_t nb_mantissa_44[44] = {
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  2,
     2,  2,  2,  2,  2,  3,  3,  3,  3,  4,  4,  5,  5,  6,  7,  7,
     8,  9, 10, 11, 12, 13, 15, 16, 18, 20, 22, 25,
};

static const uint8_t nb_mantissa_50[50] = {
     1,  1,  1,  1,  1,  1,  1,  1,  2,  2,  2,  2,  2,  3,  3,  3,
     3,  4,  4,  5,  5,  6,  6,  7,  8,  9,  9, 10, 12, 13, 14, 16,
    18, 19, 22, 24, 27, 29, 32, 36, 40, 44, 49, 54, 60, 66, 74, 82,
    90, 100,
};

static const uint8_t imdct_bits_tab[3] = { 8, 9, 11 };

static const DBEGroup grp_tab_0[1] = {
    { 50, { 27, 23 }, 0, 0, nb_mantissa_50, 2, 0, 1152, 0, 1408, 0 },
};

static const DBEGroup grp_tab_1[8] = {
    { 38, { 12, 26 }, 0, 0, nb_mantissa_38, 0, 0, 192, 0, 256, 0 },
    { 38, { 12, 26 }, 38, 128, nb_mantissa_38, 0, 1, 256, 64, 448, 0 },
    { 38, { 12, 26 }, 76, 256, nb_mantissa_38, 0, 1, 256, 192, 704, 0 },
    { 38, { 12, 26 }, 114, 384, nb_mantissa_38, 0, 1, 256, 320, 0, 0 },
    { 38, { 12, 26 }, 152, 512, nb_mantissa_38, 0, 1, 256, 448, 0, 0 },
    { 38, { 12, 26 }, 190, 640, nb_mantissa_38, 0, 1, 256, 576, 0, 0 },
    { 38, { 12, 26 }, 228, 768, nb_mantissa_38, 0, 1, 256, 704, 0, 0 },
    { 38, { 12, 26 }, 266, 896, nb_mantissa_38, 0, 1, 256, 832, 0, 0 },
};

static const DBEGroup grp_tab_2[7] = {
    { 38, { 12, 26 }, 0, 0, nb_mantissa_38, 0, 0, 192, 0, 256, 0 },
    { 38, { 12, 26 }, 38, 128, nb_mantissa_38, 0, 1, 256, 64, 448, 0 },
    { 38, { 12, 26 }, 76, 256, nb_mantissa_38, 0, 1, 256, 192, 704, 0 },
    { 38, { 12, 26 }, 114, 384, nb_mantissa_38, 0, 1, 256, 320, 0, 0 },
    { 38, { 12, 26 }, 152, 512, nb_mantissa_38, 0, 1, 256, 448, 0, 0 },
    { 38, { 12, 26 }, 190, 640, nb_mantissa_38, 0, 1, 256, 576, 0, 0 },
    { 44, { 19, 25 }, 228, 768, nb_mantissa_44, 1, 1, 448, 704, 960, 64 },
};

static const DBEGroup grp_tab_3[1] = {
    { 21, { 21 }, 0, 0, nb_mantissa_50, 2, 0, 1152, 0, 1408, 0 },
};

static const DBEGroup grp_tab_4[1] = {
    { 50, { 27, 23 }, 0, 0, nb_mantissa_50, 2, 2, 1152, 0, 1408, 896 },
};

static const DBEGroup grp_tab_5[8] = {
    { 38, { 12, 26 }, 0, 0, nb_mantissa_38, 0, 1, 256, 64, 0, 0 },
    { 38, { 12, 26 }, 38, 128, nb_mantissa_38, 0, 1, 256, 192, 0, 0 },
    { 38, { 12, 26 }, 76, 256, nb_mantissa_38, 0, 1, 256, 320, 0, 0 },
    { 38, { 12, 26 }, 114, 384, nb_mantissa_38, 0, 1, 256, 448, 0, 0 },
    { 38, { 12, 26 }, 152, 512, nb_mantissa_38, 0, 1, 256, 576, 0, 0 },
    { 38, { 12, 26 }, 190, 640, nb_mantissa_38, 0, 1, 256, 704, 3008, 0 },
    { 38, { 12, 26 }, 228, 768, nb_mantissa_38, 0, 1, 256, 832, 2752, 0 },
    { 38, { 12, 26 }, 266, 896, nb_mantissa_38, 0, 2, 192, 960, 2560, 64 },
};

static const DBEGroup grp_tab_6[7] = {
    { 44, { 19, 25 }, 0, 0, nb_mantissa_44, 1, 1, 448, 0, 3264, 0 },
    { 38, { 12, 26 }, 44, 256, nb_mantissa_38, 0, 1, 256, 320, 0, 0 },
    { 38, { 12, 26 }, 82, 384, nb_mantissa_38, 0, 1, 256, 448, 0, 0 },
    { 38, { 12, 26 }, 120, 512, nb_mantissa_38, 0, 1, 256, 576, 0, 0 },
    { 38, { 12, 26 }, 158, 640, nb_mantissa_38, 0, 1, 256, 704, 3008, 0 },
    { 38, { 12, 26 }, 196, 768, nb_mantissa_38, 0, 1, 256, 832, 2752, 0 },
    { 38, { 12, 26 }, 234, 896, nb_mantissa_38, 0, 2, 192, 960, 2560, 64 },
};

static const DBEGroup grp_tab_7[1] = {
    { 21, { 21 }, 0, 0, nb_mantissa_50, 2, 2, 1152, 0, 1408, 896 },
};

static const DBEGroup *const frm_ofs_tab[2][4] = {
    { grp_tab_0, grp_tab_1, grp_tab_2, grp_tab_3 },
    { grp_tab_4, grp_tab_5, grp_tab_6, grp_tab_7 }
};

static const uint8_t mantissa_size1[16][4] = {
    {  0,  0,  0,  0 }, {  2,  1,  1,  1 }, {  3,  2,  1,  1 }, {  4,  3,  2,  1 },
    {  5,  4,  3,  2 }, {  6,  5,  4,  3 }, {  7,  6,  5,  4 }, {  8,  7,  6,  5 },
    {  9,  8,  7,  6 }, { 10,  9,  8,  7 }, { 11, 10,  9,  8 }, { 12, 11, 10,  9 },
    { 13, 12, 11, 10 }, { 14, 13, 12, 11 }, { 15, 14, 13, 12 }, { 16, 15, 14, 13 },
};

static const uint8_t mantissa_size2[16][4] = {
    {  0,  0,  0,  0 }, {  2,  1,  2,  2 }, {  3,  2,  3,  3 }, {  4,  3,  4,  4 },
    {  5,  4,  5,  5 }, {  6,  5,  6,  6 }, {  7,  6,  7,  7 }, {  8,  7,  8,  8 },
    {  9,  8,  9,  9 }, { 10,  9, 10, 10 }, { 11, 10, 11, 11 }, { 12, 11, 12, 12 },
    { 13, 12, 13, 13 }, { 14, 13, 14, 14 }, { 15, 14, 15, 15 }, { 16, 15, 16, 16 },
};

static const float start_window[192] = {
    0.00161569379826, 0.00185748233347, 0.00198562758548, 0.00207834078104,
    0.00215717748523, 0.00223067096393, 0.00230299213147, 0.00237651215396,
    0.00245275561606, 0.00253281402069, 0.00261754673613, 0.00270768786168,
    0.00280390761895, 0.00290684998656, 0.00301715751161, 0.00313548872798,
    0.00326253122934, 0.00339901215995, 0.00354570716636, 0.00370344845023,
    0.00387313232586, 0.00405572653911, 0.00425227750970, 0.00446391759265,
    0.00469187240551, 0.00493746822816, 0.00520213944619, 0.00548743597507,
    0.00579503056737, 0.00612672586953, 0.00648446105606, 0.00687031782873,
    0.00728652552677, 0.00773546505205, 0.00821967127415, 0.00874183354619,
    0.00930479393832, 0.00991154278653, 0.01056521116692, 0.01126905994567,
    0.01202646513050, 0.01284089936559, 0.01371590957417, 0.01465509096066,
    0.01566205783408, 0.01674041199523, 0.01789370972358, 0.01912542867865,
    0.02043893626265, 0.02183746113793, 0.02332406961796, 0.02490164852364,
    0.02657289580178, 0.02834031974193, 0.03020624702903, 0.03217283918354,
    0.03424211623810, 0.03641598586180, 0.03869627565015, 0.04108476601498,
    0.04358322107390, 0.04619341515939, 0.04891715301882, 0.05175628239149,

    0.05471237327267, 0.05778734733755, 0.06098291402413, 0.06430101352084,
    0.06774345212186, 0.07131188644726, 0.07500780649199, 0.07883251748595,
    0.08278712056651, 0.08687249228061, 0.09108926295730, 0.09543779401074,
    0.09991815425851, 0.10453009536427, 0.10927302653894, 0.11414598865987,
    0.11914762799220, 0.12427616972097, 0.12952939152560, 0.13490459744934,
    0.14039859233595, 0.14600765712201, 0.15172752528722, 0.15755336077528,
    0.16347973770491, 0.16950062219342, 0.17560935661442, 0.18179864660619,
    0.18806055113821, 0.19438647593012, 0.20076717050010, 0.20719272909882,
    0.21365259576030, 0.22013557367283, 0.22662983904194, 0.23312295958328,
    0.23960191774666, 0.24605313873388, 0.25246252333253, 0.25881548554631,
    0.26509699495987, 0.27129162373316, 0.27738359807707, 0.28335685401987,
    0.28919509723179, 0.29488186663467, 0.30040060148455, 0.30573471157819,
    0.31086765019993, 0.31578298939317, 0.32046449711227, 0.32489621578468,
    0.32906254179156, 0.33294830535654, 0.33653885031840, 0.33982011325336,
    0.34277870140679, 0.34540196889300, 0.34767809062480, 0.34959613344194,
    0.35114612391958, 0.35231911235422, 0.35310723244504, 0.35350375621308,

    0.35350314372945, 0.35310108725579, 0.35229454943591, 0.35108179521634,
    0.34946241721522, 0.34743735430290, 0.34500890320420, 0.34218072298001,
    0.33895783229541, 0.33534659943168, 0.33135472505060, 0.32699121776996,
    0.32226636266000, 0.31719168282019, 0.31177989424432, 0.30604485422875,
    0.30000150362379, 0.29366580327088, 0.28705466500775, 0.28018587766131,
    0.27307802848095, 0.26575042049535, 0.25822298630189, 0.25051619882000,
    0.24265097955783, 0.23464860495522, 0.22653061137548, 0.21831869932335,
    0.21003463746705, 0.20170016703857, 0.19333690717811, 0.18496626177620,
    0.17660932835062, 0.16828680947474, 0.16001892724986, 0.15182534128597,
    0.14372507062477, 0.13573642000364, 0.12787691082233, 0.12016321713317,
    0.11261110693234, 0.10523538898282, 0.09804986534955, 0.09106728977263,
    0.08429933194438, 0.07775654768810, 0.07144835495683, 0.06538301547324,
    0.05956762170687, 0.05400808871425, 0.04870915012107, 0.04367435714993,
    0.03890607899172, 0.03440550179663, 0.03017262174627, 0.02620622428513,
    0.02250383492507, 0.01906161305732, 0.01587412848221, 0.01293388032354,
    0.01023019677288, 0.00774641320626, 0.00545109736891, 0.00325868651263,
};

static const float short_window2[192] = {
    0.00018861094606, 0.00033433010202, 0.00050309624485, 0.00070306161748,
    0.00093995174533, 0.00121913067128, 0.00154606505568, 0.00192647806126,
    0.00236641248692, 0.00287225985240, 0.00345077377440, 0.00410907465023,
    0.00485464855241, 0.00569534163219, 0.00663935063508, 0.00769520981249,
    0.00887177436246, 0.01017820046395, 0.01162392194150, 0.01321862359335,
    0.01497221122468, 0.01689477844427, 0.01899657030441, 0.02128794388846,
    0.02377932597692, 0.02648116795039, 0.02940389811590, 0.03255787167130,
    0.03595331854986, 0.03960028941437, 0.04350860009563, 0.04768777479454,
    0.05214698838949, 0.05689500821121, 0.06194013566525, 0.06729014809766,
    0.07295224131210, 0.07893297315602, 0.08523820859989, 0.09187306673620,
    0.09884187012422, 0.10614809690222, 0.11379433608064, 0.12178224641797,
    0.13011251926531, 0.13878484574660, 0.14779788861830, 0.15714925912610,
    0.16683549914631, 0.17685206886673, 0.18719334022589, 0.19785259629099,
    0.20882203671372, 0.22009278936030, 0.23165492816694, 0.24349749722585,
    0.25560854105961, 0.26797514099368, 0.28058345748882, 0.29341877824732,
    0.30646557185942, 0.31970754671026, 0.33312771482295, 0.34670846027024,

    0.36043161174692, 0.37427851885723, 0.38823013163645, 0.40226708279486,
    0.41636977214436, 0.43051845264462, 0.44469331748632, 0.45887458761470,
    0.47304259908636, 0.48717788964798, 0.50126128392546, 0.51527397661778,
    0.52919761310050, 0.54301436685998, 0.55670701320069, 0.57025899869448,
    0.58365450587230, 0.59687851269542, 0.60991684638414, 0.62275623122793,
    0.63538433005035, 0.64778977905593, 0.65996221584264, 0.67189230042379,
    0.68357172916486, 0.69499324160511, 0.70615062019861, 0.71703868307548,
    0.72765326998919, 0.73799122168099, 0.74805035295521, 0.75782941981995,
    0.76732808110520, 0.77654685502339, 0.78548707118622, 0.79415081863423,
    0.80254089047207, 0.81066072573188, 0.81851434910893, 0.82610630922734,
    0.83344161609862, 0.84052567843230, 0.84736424144524, 0.85396332579459,
    0.86032916822973, 0.86646816451999, 0.87238681516918, 0.87809167437532,
    0.88358930263537, 0.88888622333073, 0.89398888356256, 0.89890361943564,
    0.90363662591861, 0.90819393133744, 0.91258137648979, 0.91680459830070,
    0.92086901787718, 0.92477983276087, 0.92854201312583, 0.93216030163834,
    0.93563921662343, 0.93898305819384, 0.94219591693690, 0.94528168477979,

    0.94823843319821, 0.95106834367330, 0.95377776558539, 0.95636718335775,
    0.95883679961479, 0.96118650212341, 0.96341583179195, 0.96552395212906,
    0.96750962060547, 0.96937116231768, 0.97110644638309, 0.97271286544154,
    0.97418731862798, 0.97552619834964, 0.97672538116257, 0.97778022299974,
    0.97868555895586, 0.97943570778357, 0.98002448120255, 0.98044519806866,
    0.98069070339493, 0.98075339216123, 0.98062523779637, 0.98029782516478,
    0.97976238784222, 0.97900984942031, 0.97803086854002, 0.97681588731895,
    0.97535518280755, 0.97363892108474, 0.97165721358452, 0.96940017523145,
    0.96685798395452, 0.96402094114589, 0.96087953263194, 0.95742448973047,
    0.95364684997699, 0.94953801711660, 0.94508981997396, 0.94029456983253,
    0.93514511597504, 0.92963489905951, 0.92375800202883, 0.91750919827624,
    0.91088399681406, 0.90387868421832, 0.89649036314692, 0.88871698725397,
    0.88055739234735, 0.87201132366062, 0.86307945913336, 0.85376342861693,
    0.84406582894455, 0.83399023482637, 0.82354120554757, 0.81272428745995,
    0.80154601230457, 0.79001389138101, 0.77813640562199, 0.76592299164227,
    0.75338402384395, 0.74053079267526, 0.72737547915460, 0.71393112578527,
};

static const float short_window3[64] = {
    0.00326887936450, 0.00550242900936, 0.00786846643791, 0.01045683453520,
    0.01330402120132, 0.01643221072863, 0.01985798040609, 0.02359509464766,
    0.02765559221954, 0.03205025893128, 0.03678884369614, 0.04188015679495,
    0.04733210987781, 0.05315172583924, 0.05934513287609, 0.06591755045290,
    0.07287327156378, 0.08021564389822, 0.08794705152307, 0.09606889811179,
    0.10458159240070, 0.11348453632940, 0.12277611617809, 0.13245369691511,
    0.14251361989876, 0.15295120402567, 0.16376075037904, 0.17493555039885,
    0.18646789757072, 0.19834910260891, 0.21056951208995, 0.22311853047787,
    0.23598464546683, 0.24915545655419, 0.26261770674500, 0.27635731727778,
    0.29035942525136, 0.30460842402318, 0.31908800624032, 0.33378120935681,
    0.34867046348260, 0.36373764140285, 0.37896411059909, 0.39433078709788,
    0.40981819096657, 0.42540650327031, 0.44107562429959, 0.45680523287270,
    0.47257484651351, 0.48836388230077, 0.50415171818214, 0.51991775454258,
    0.53564147581496, 0.55130251191887, 0.56688069931047, 0.58235614142007,
    0.59770926827271, 0.61292089506118, 0.62797227945823, 0.64284517745255,
    0.65752189749349, 0.67198535273209, 0.68621911114984, 0.70020744337099,
};

static const uint8_t dc_code_tab[5] = { 0, 0, 0, 1, 1 };

static const uint8_t ht_code_tab[5] = { 0, 0, 1, 2, 2 };

static const uint8_t band_ofs_tab[3][4] = {
    { 12, 8, 4, 0 }, { 14, 10, 6, 0 }, { 12, 8, 4, 0 }
};

static const uint8_t band_low_tab[3] = { 9, 17, 24 };

static const uint16_t fast_gain_tab[8] = {
    128, 256, 384, 512, 640, 768, 896, 1024
};

static const uint16_t slow_decay_tab[2][2] = { { 27, -1 }, { 32, 21 } };

static const uint16_t misc_decay_tab[3][2][2] = {
    { { 354, -1 }, { 425, 425 } },
    { { 266, -1 }, { 320,  -1 } },
    { { 213, -1 }, { 256,  -1 } }
};

static const uint16_t fast_decay_tab[3][2][2][50] = {
    {{{
        142, 142, 142, 142, 142, 142, 142, 142, 142, 142,
        142, 142, 142, 142, 142, 142, 142, 142, 142, 142,
        142, 142, 142, 142, 142, 142, 142, 142, 142, 142,
        142, 142, 142, 142, 142, 142, 142, 142,
    }, {
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
    }}, {{
        170, 170, 170, 170, 170, 170, 170, 170, 170, 170,
        170, 170, 170, 170, 170, 170, 170, 170, 170, 170,
        170, 170, 170, 170, 170, 170, 170, 170, 170, 170,
        170, 170, 170, 170, 170, 170, 170, 170,
    }, {
         64,  64,  64,  64,  64,  64,  64,  64,  64,  64,
         64,  64,  64,  64,  64,  64,  64,  64,  64,  64,
         64,  64,  64,  64,  64,  64,  64,  64,  64,  64,
         64,  64,  64,  64,  64,  64,  64,  64,
    }}}, {{{
        266, 266, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106,
    }, {
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,
    }}, {{
        319, 319, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128,
    }, {
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,
    }}}, {{{
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
        106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
    }, {
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
    }}, {{
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
        128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
    }, {
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
         -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
    }}}
};

static const uint16_t fast_gain_adj_tab[3][2][62] = {
    {{
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   1,   2,   4,   7,  11,  16,  29,  44,  59,
         76,  94, 116, 142, 179, 221, 252, 285, 312, 334,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          2,   5,   8,  10,  15,  28,  42,  57,  75,  93,
        115, 140, 177, 219, 247, 280, 308, 330, 427, 533,
    }}, {{
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   2,   5,   8,  12,  21,  35,  51,  69,  89,
        111, 138, 176, 220, 251, 284, 312, 334,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   2,
          5,   8,  11,  18,  33,  49,  65,  84, 106, 132,
        168, 214, 245, 279, 308, 329, 427, 533,
    }}, {{
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   1,   4,   7,  10,  17,
         31,  47,  65,  84, 107, 134, 171, 215, 250, 283,
        312, 334,
    }, {
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
          0,   0,   0,   0,   3,   6,   9,  13,  27,  43,
         60,  79, 100, 126, 160, 207, 242, 276, 307, 329,
        427, 533,
    }}
};

static const uint16_t slow_gain_tab[3][2][50] = {
    {{
        3072, 3072, 3072, 3072, 3072, 3072, 1063, 1063, 1063, 1063,
        1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063,
        1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063,
        1063, 1063, 1063, 1063, 1063, 1063, 1063, 1063,
    }, {
        3072, 3072, 3072, 3072, 3072, 3072,  850,  850,  850,  850,
         850,  850,  850,  850,  850,  850,  850,  850,  850,  850,
         850,  850,  850,  850,  850,  850,  850,  850,  850,  850,
         850,  850,  850,  850,  850,  850,  850,  850,
    }}, {{
        3072, 1212, 1212, 1212,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,
    }, {
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,
    }}, {{
        3072, 3072, 3072, 3072, 3072, 3072, 3072, 3072, 3072, 3072,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
         999,  999,  999,  999,  999,  999,  999,  999,  999,  999,
    }, {
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
          -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
    }}
};

static const uint16_t hearing_thresh_tab[3][3][50] = {
    {{
        1403, 1141, 1000,  959,  948,  957,  946,  925,  899,  871,
         843,  815,  789,  766,  745,  727,  705,  687,  681,  686,
         701,  725,  768,  854,  940, 1018, 1075, 1103, 1111, 1106,
        1098, 1105, 1142, 1237, 1419, 1721, 2169, 2805,
    }, {
        1401, 1130,  995,  957,  947,  955,  941,  918,  890,  861,
         831,  803,  777,  754,  734,  717,  698,  684,  682,  692,
         712,  743,  798,  894,  976, 1045, 1091, 1109, 1110, 1102,
        1098, 1116, 1174, 1300, 1526, 1884, 2401, 3072,
    }, {
        1393, 1086,  974,  949,  957,  941,  913,  878,  843,  808,
         777,  750,  727,  708,  695,  686,  681,  689,  714,  752,
         811,  888,  971, 1044, 1087, 1108, 1110, 1102, 1098, 1115,
        1172, 1290, 1489, 1812, 2293, 2964, 3072, 3072,
    }}, {{
        1412, 1343, 1141, 1047, 1000,  974,  959,  951,  948,  947,
         957,  953,  946,  936,  925,  906,  878,  850,  822,  795,
         771,  745,  719,  700,  687,  681,  685,  701,  733,  784,
         885,  977, 1047, 1092, 1110, 1108, 1099, 1102, 1138, 1233,
        1413, 1711, 2157, 2797,
    }, {
        1412, 1336, 1130, 1040,  995,  970,  957,  950,  947,  947,
         955,  950,  941,  930,  918,  897,  868,  838,  810,  783,
         759,  734,  710,  693,  684,  681,  690,  712,  752,  823,
         924, 1009, 1069, 1102, 1111, 1104, 1098, 1111, 1168, 1295,
        1518, 1873, 2388, 3072,
    }, {
        1411, 1293, 1086, 1009,  974,  957,  949,  947,  957,  951,
         941,  928,  913,  896,  878,  852,  817,  785,  756,  732,
         713,  695,  683,  682,  689,  710,  746,  811,  906,  992,
        1061, 1099, 1111, 1106, 1098, 1107, 1155, 1266, 1471, 1799,
        2277, 2945, 3072, 3072,
    }}, {{
        1431, 1412, 1403, 1379, 1343, 1293, 1229, 1180, 1125, 1075,
        1040, 1014,  996,  979,  965,  957,  951,  948,  947,  957,
         951,  940,  924,  903,  877,  846,  815,  785,  753,  725,
         702,  686,  681,  689,  714,  760,  847,  947, 1028, 1083,
        1108, 1109, 1101, 1100, 1132, 1222, 1402, 1705, 2160, 2803,
    }, {
        1431, 1412, 1401, 1375, 1336, 1278, 1215, 1168, 1115, 1066,
        1032, 1008,  991,  975,  962,  954,  950,  947,  947,  955,
         948,  935,  916,  894,  866,  835,  803,  772,  742,  715,
         695,  683,  683,  697,  729,  784,  887,  982, 1054, 1096,
        1111, 1106, 1098, 1107, 1159, 1281, 1505, 1865, 2391, 3072,
    }, {
        1427, 1411, 1393, 1353, 1293, 1215, 1160, 1118, 1072, 1031,
        1003,  984,  971,  960,  952,  948,  947,  957,  952,  941,
         924,  902,  876,  847,  815,  781,  750,  723,  700,  685,
         681,  691,  719,  766,  858,  958, 1039, 1089, 1109, 1108,
        1099, 1102, 1141, 1245, 1442, 1766, 2250, 2930, 3072, 3072,
    }}
};

static const int16_t lwc_gain_tab[11][7] = {
    {   -21,  -197,  -271,  -466, 32767, 32767, 32767 },
    {  -197,   -29,  -244,  -271,  -540, 32767, 32767 },
    {  -271,  -244,   -29,  -249,  -271,  -593, 32767 },
    {  -466,  -271,  -249,   -29,  -251,  -271,  -632 },
    {  -540,  -271,  -251,   -29,  -251,  -271,  -664 },
    {  -593,  -271,  -251,   -29,  -252,  -271,  -690 },
    {  -632,  -271,  -252,   -29,  -252,  -271,  -711 },
    {  -664,  -271,  -252,   -29,  -252,  -271,  -730 },
    {  -690,  -271,  -252,   -29,  -252,  -271,  -745 },
    {  -711,  -271,  -252,   -29,  -253,  -271,  -759 },
    {  -730,  -271,  -253,   -29,  -253,  -271,  -771 },
};

static const int16_t lwc_adj_tab[7] = {
    -192, -320, -448, -512, -448, -320, -192,
};

static const uint8_t log_add_tab[212] = {
    64, 63, 62, 61, 60, 59, 58, 57, 56, 55, 54, 53, 52, 52, 51, 50,
    49, 48, 47, 47, 46, 45, 44, 44, 43, 42, 41, 41, 40, 39, 38, 38,
    37, 36, 36, 35, 35, 34, 33, 33, 32, 32, 31, 30, 30, 29, 29, 28,
    28, 27, 27, 26, 26, 25, 25, 24, 24, 23, 23, 22, 22, 21, 21, 21,
    20, 20, 19, 19, 19, 18, 18, 18, 17, 17, 17, 16, 16, 16, 15, 15,
    15, 14, 14, 14, 13, 13, 13, 13, 12, 12, 12, 12, 11, 11, 11, 11,
    10, 10, 10, 10, 10,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,
     7,  7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  6,  6,  5,  5,
     5,  5,  5,  5,  5,  5,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
     4,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  2,
     2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
     2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  0,  0,
};

static const uint8_t bap_tab[64] = {
     0,  1,  1,  1,  1,  1,  2,  2,  2,  2,  2,  3,  3,  3,  3,  4,
     4,  4,  4,  5,  5,  5,  5,  6,  6,  6,  6,  7,  7,  7,  7,  8,
     8,  8,  8,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11, 11, 12,
    12, 12, 12, 13, 13, 13, 13, 14, 14, 14, 14, 15, 15, 15, 15, 15,
};

static float mantissa_tab1[17][4];
static float mantissa_tab2[17][4];
static float mantissa_tab3[17][4];
static float exponent_tab[50];
static float gain_tab[1024];

DECLARE_ALIGNED(32, static float, window)[3712];

static int skip_input(DBEContext *s, int nb_words)
{
    if (nb_words > s->input_size) {
        av_log(s->avctx, AV_LOG_ERROR, "Packet too short\n");
        return AVERROR_INVALIDDATA;
    }

    s->input      += nb_words * s->word_bytes;
    s->input_size -= nb_words;
    return 0;
}

static int parse_key(DBEContext *s)
{
    if (s->key_present) {
        const uint8_t *key = s->input;
        int      ret = skip_input(s, 1);
        if (ret < 0)
            return ret;
        return AV_RB24(key) >> 24 - s->word_bits;
    }
    return 0;
}

static int parse_metadata_ext(DBEDecodeContext *s1)
{
    DBEContext *s = &s1->dectx;
    if (s->metadata.mtd_ext_size)
        return skip_input(s, s->key_present + s->metadata.mtd_ext_size + 1);
    return 0;
}

static void unbias_exponents(DBEContext *s, DBEChannel *c, DBEGroup *g)
{
    int mstr_exp[MAX_MSTR_EXP];
    int bias_exp[MAX_BIAS_EXP];
    int i, j, k;

    for (i = 0; i < c->nb_mstr_exp; i++)
        mstr_exp[i] = get_bits(&s->gb, 2) * 6;

    for (i = 0; i < g->nb_exponent; i++)
        bias_exp[i] = get_bits(&s->gb, 5);

    for (i = k = 0; i < c->nb_mstr_exp; i++)
        for (j = 0; j < g->nb_bias_exp[i]; j++, k++)
            c->exponents[g->exp_ofs + k] = mstr_exp[i] + bias_exp[k];
}

static int parse_exponents(DBEContext *s, DBEChannel *c)
{
    DBEGroup *p, *g;
    int i;

    for (i = 0, p = NULL, g = c->groups; i < c->nb_groups; i++, p = g, g++) {
        c->exp_strategy[i] = !i || g->nb_exponent != p->nb_exponent || get_bits1(&s->gb);
        if (c->exp_strategy[i]) {
            unbias_exponents(s, c, g);
        } else {
            memcpy(c->exponents + g->exp_ofs,
                   c->exponents + p->exp_ofs,
                   g->nb_exponent * sizeof(c->exponents[0]));
        }
    }

    return 0;
}

static inline int log_add(int a, int b)
{
    int c = FFABS(a - b) >> 1;
    return FFMAX(a, b) + log_add_tab[FFMIN(c, 211)];
}

static void calc_lowcomp(int *msk_val)
{
    int lwc_val[17] = { 0 };
    int i, j, k;

    for (i = 0; i < 11; i++) {
        int max_j = 0;
        int max_v = INT_MIN;
        int thr   = 0;

        for (j = FFMAX(i - 3, 0), k = 0; j <= i + 3; j++, k++) {
            int v = msk_val[j] + lwc_gain_tab[i][k];
            if (v > max_v) {
                max_j = j;
                max_v = v;
            }
            thr = log_add(thr, v);
        }

        if (msk_val[i] < thr) {
            for (j = FFMAX(max_j - 3, 0),
                 k = FFMAX(3 - max_j, 0);
                 j <= max_j + 3; j++, k++)
                lwc_val[j] += lwc_adj_tab[k];
        }
    }

    for (i = 0; i < 16; i++) {
        int v = FFMAX(lwc_val[i], -512);
        msk_val[i] = FFMAX(msk_val[i] + v, 0);
    }
}

static void bit_allocate(int nb_exponent, int nb_code, int fr_code,
                         int *exp, int *bap,
                         int fg_spc, int fg_ofs, int msk_mod, int snr_ofs)
{
    int msk_val[MAX_BIAS_EXP];
    int psd_val[MAX_BIAS_EXP];
    int fast_leak  = 0;
    int slow_leak  = 0;
    int dc_code    = dc_code_tab[fr_code - 1];
    int ht_code    = ht_code_tab[fr_code - 1];
    int fast_gain  = fast_gain_tab[fg_ofs];
    int slow_decay = slow_decay_tab[dc_code][msk_mod];
    int misc_decay = misc_decay_tab[nb_code][dc_code][msk_mod];
    const uint16_t *slow_gain      = slow_gain_tab[nb_code][msk_mod];
    const uint16_t *fast_decay     = fast_decay_tab[nb_code][dc_code][msk_mod];
    const uint16_t *fast_gain_adj  = fast_gain_adj_tab[nb_code][dc_code];
    const uint16_t *hearing_thresh = hearing_thresh_tab[nb_code][ht_code];
    int i;

    for (i = 0; i < nb_exponent; i++)
        psd_val[i] = (48 - exp[i]) * 64;

    fast_gain_adj += band_ofs_tab[nb_code][fg_spc];
    for (i = 0; i < nb_exponent; i++) {
        fast_leak = log_add(fast_leak  - fast_decay[i],
                            psd_val[i] - fast_gain + fast_gain_adj[i]);
        slow_leak = log_add(slow_leak  - slow_decay,
                            psd_val[i] - slow_gain[i]);
        msk_val[i] = FFMAX(fast_leak, slow_leak);
    }

    fast_leak = 0;
    for (i = nb_exponent - 1; i > band_low_tab[nb_code]; i--) {
        fast_leak = log_add(fast_leak - misc_decay, psd_val[i] - fast_gain);
        msk_val[i] = FFMAX(msk_val[i], fast_leak);
    }

    for (i = 0; i < nb_exponent; i++)
        msk_val[i] = FFMAX(msk_val[i], hearing_thresh[i]);

    if (!nb_code)
        calc_lowcomp(msk_val);

    for (i = 0; i < nb_exponent; i++) {
        int v = 16 * (snr_ofs - 64) + psd_val[i] - msk_val[i] >> 5;
        bap[i] = bap_tab[av_clip_uintp2(v, 6)];
    }
}

static int parse_bit_alloc(DBEDecodeContext *s1, DBEChannel *c)
{
    DBEContext *s = &s1->dectx;
    DBEGroup *p, *g;
    int bap_strategy[MAX_GROUPS], fg_spc[MAX_GROUPS];
    int fg_ofs[MAX_GROUPS], msk_mod[MAX_GROUPS];
    int i, snr_ofs;

    for (i = 0; i < c->nb_groups; i++) {
        bap_strategy[i] = !i || get_bits1(&s->gb);
        if (bap_strategy[i]) {
             fg_spc[i] = get_bits(&s->gb, 2);
             fg_ofs[i] = get_bits(&s->gb, 3);
            msk_mod[i] = get_bits1(&s->gb);
        } else {
             fg_spc[i] =  fg_spc[i - 1];
             fg_ofs[i] =  fg_ofs[i - 1];
            msk_mod[i] = msk_mod[i - 1];
        }
    }

    if (get_bits1(&s->gb)) {
        avpriv_report_missing_feature(s->avctx, "Delta bit allocation");
        return AVERROR_PATCHWELCOME;
    }

    snr_ofs = get_bits(&s->gb, 8);
    if (!snr_ofs) {
        memset(c->bap, 0, sizeof(c->bap));
        return 0;
    }

    for (i = 0, p = NULL, g = c->groups; i < c->nb_groups; i++, p = g, g++) {
        if (c->exp_strategy[i] || bap_strategy[i]) {
            bit_allocate(g->nb_exponent, g->imdct_idx, s->metadata.fr_code,
                         c->exponents + g->exp_ofs, c->bap + g->exp_ofs,
                         fg_spc[i], fg_ofs[i], msk_mod[i], snr_ofs);
        } else {
            memcpy(c->bap + g->exp_ofs,
                   c->bap + p->exp_ofs,
                   g->nb_exponent * sizeof(c->bap[0]));
        }
    }

    return 0;
}

static int parse_indices(DBEContext *s, DBEChannel *c)
{
    DBEGroup *p, *g;
    int i, j;

    for (i = 0, p = NULL, g = c->groups; i < c->nb_groups; i++, p = g, g++) {
        if (get_bits1(&s->gb)) {
            int start = get_bits(&s->gb, 6);

            if (start > g->nb_exponent) {
                av_log(s->avctx, AV_LOG_ERROR, "Invalid start index\n");
                return AVERROR_INVALIDDATA;
            }

            for (j = 0; j < start; j++)
                c->idx[g->exp_ofs + j] = 0;

            for (; j < g->nb_exponent; j++)
                c->idx[g->exp_ofs + j] = get_bits(&s->gb, 2);
        } else if (i && g->nb_exponent == p->nb_exponent) {
            memcpy(c->idx + g->exp_ofs,
                   c->idx + p->exp_ofs,
                   g->nb_exponent * sizeof(c->idx[0]));
        } else {
            memset(c->idx + g->exp_ofs, 0, g->nb_exponent * sizeof(c->idx[0]));
        }
    }

    return 0;
}

static int parse_mantissas(DBEContext *s, DBEChannel *c)
{
    DBEGroup *g;
    int i, j, k;

    for (i = 0, g = c->groups; i < c->nb_groups; i++, g++) {
        float *mnt = c->mantissas + g->mnt_ofs;

        for (j = 0; j < g->nb_exponent; j++) {
            int bap     = c->bap[g->exp_ofs + j];
            int idx     = c->idx[g->exp_ofs + j];
            int size1   = mantissa_size1[bap][idx];
            int count   = g->nb_mantissa[j];
            float exp   = exponent_tab[c->exponents[g->exp_ofs + j]];
            float scale = mantissa_tab1[size1][idx] * exp;

            if (!size1) {
                memset(mnt, 0, count * sizeof(*mnt));
            } else if (idx) {
                int values[100];
                int escape = -(1 << size1 - 1);

                for (k = 0; k < count; k++)
                    values[k] = get_sbits(&s->gb, size1);

                for (k = 0; k < count; k++) {
                    if (values[k] != escape) {
                        mnt[k] = values[k] * scale;
                    } else {
                        int size2 = mantissa_size2[bap][idx];
                        int value = get_sbits(&s->gb, size2);
                        float a = mantissa_tab2[size2][idx];
                        float b = mantissa_tab3[size2][idx];
                        if (value < 0)
                            mnt[k] = ((value + 1) * a - b) * exp;
                        else
                            mnt[k] = (value * a + b) * exp;
                    }
                }
            } else {
                for (k = 0; k < count; k++)
                    mnt[k] = get_sbits(&s->gb, size1) * scale;
            }

            mnt += count;
        }

        for (; j < g->nb_exponent + c->bw_code; j++) {
            memset(mnt, 0, g->nb_mantissa[j] * sizeof(*mnt));
            mnt += g->nb_mantissa[j];
        }
    }

    return 0;
}

static int parse_channel(DBEDecodeContext *s1, int ch, int seg_id)
{
    DBEContext *s = &s1->dectx;
    DBEChannel *c = &s1->channels[seg_id][ch];
    int i, ret;

    if (s->metadata.rev_id[ch] > 1) {
        avpriv_report_missing_feature(s->avctx, "Encoder revision %d", s->metadata.rev_id[ch]);
        return AVERROR_PATCHWELCOME;
    }

    if (ch == lfe_channel_tab[s->metadata.prog_conf]) {
        c->gr_code = 3;
        c->bw_code = 29;
    } else {
        c->gr_code = get_bits(&s->gb, 2);
        c->bw_code = get_bits(&s->gb, 3);
        if (c->gr_code == 3) {
            av_log(s->avctx, AV_LOG_ERROR, "Invalid group type code\n");
            return AVERROR_INVALIDDATA;
        }
    }

    c->nb_groups   = nb_groups_tab[c->gr_code];
    c->nb_mstr_exp = nb_mstr_exp_tab[c->gr_code];

    for (i = 0; i < c->nb_groups; i++) {
        c->groups[i] = frm_ofs_tab[seg_id][c->gr_code][i];
        if (c->nb_mstr_exp == 2) {
            c->groups[i].nb_exponent    -= c->bw_code;
            c->groups[i].nb_bias_exp[1] -= c->bw_code;
        }
    }

    if ((ret = parse_exponents(s, c)) < 0)
        return ret;
    if ((ret = parse_bit_alloc(s1, c)) < 0)
        return ret;
    if ((ret = parse_indices(s, c)) < 0)
        return ret;
    if ((ret = parse_mantissas(s, c)) < 0)
        return ret;

    if (get_bits_left(&s->gb) < 0) {
        av_log(s->avctx, AV_LOG_ERROR, "Read past end of channel %d\n", ch);
        return AVERROR_INVALIDDATA;
    }

    return 0;
}

static int parse_audio(DBEDecodeContext *s1, int start, int end, int seg_id)
{
    DBEContext *s = &s1->dectx;
    int ch, ret, key;

    if ((key = parse_key(s)) < 0)
        return key;

    for (ch = start; ch < end; ch++) {
        if (!s->metadata.ch_size[ch]) {
            s1->channels[seg_id][ch].nb_groups = 0;
            continue;
        }
        ret = ff_dolby_e_convert_input(s, s->metadata.ch_size[ch], key);
        if (ret < 0)
            return ret;
        if ((ret = parse_channel(s1, ch, seg_id)) < 0) {
            if (s1->avctx->err_recognition & AV_EF_EXPLODE)
                return ret;
            s1->channels[seg_id][ch].nb_groups = 0;
        }
        if ((ret = skip_input(s, s->metadata.ch_size[ch])) < 0)
            return ret;
    }

    return skip_input(s, 1);
}

static int parse_meter(DBEDecodeContext *s1)
{
    DBEContext *s = &s1->dectx;
    if (s->metadata.meter_size)
        return skip_input(s, s->key_present + s->metadata.meter_size + 1);
    return 0;
}

static void imdct_calc(DBEDecodeContext *s1, DBEGroup *g, float *result, float *values)
{
    FFTContext *imdct = &s1->imdct[g->imdct_idx];
    int n   = 1 << imdct_bits_tab[g->imdct_idx];
    int n2  = n >> 1;
    int i;

    switch (g->imdct_phs) {
    case 0:
        imdct->imdct_half(imdct, result, values);
        for (i = 0; i < n2; i++)
            result[n2 + i] = result[n2 - i - 1];
        break;
    case 1:
        imdct->imdct_calc(imdct, result, values);
        break;
    case 2:
        imdct->imdct_half(imdct, result + n2, values);
        for (i = 0; i < n2; i++)
            result[i] = -result[n - i - 1];
        break;
    default:
        av_assert0(0);
    }
}

static void transform(DBEDecodeContext *s1, DBEChannel *c, float *history, float *output)
{
    LOCAL_ALIGNED_32(float, buffer, [2048]);
    LOCAL_ALIGNED_32(float, result, [1152]);
    DBEGroup *g;
    int i;

    memset(result, 0, 1152 * sizeof(float));
    for (i = 0, g = c->groups; i < c->nb_groups; i++, g++) {
        float *src = buffer + g->src_ofs;
        float *dst = result + g->dst_ofs;
        float *win = window + g->win_ofs;

        imdct_calc(s1, g, buffer, c->mantissas + g->mnt_ofs);
        s1->fdsp->vector_fmul_add(dst, src, win, dst, g->win_len);
    }

    for (i = 0; i < 256; i++)
        output[i] = history[i] + result[i];
    for (i = 256; i < 896; i++)
        output[i] = result[i];
    for (i = 0; i < 256; i++)
        history[i] = result[896 + i];
}

static void apply_gain(DBEDecodeContext *s, int begin, int end, float *output)
{
    if (begin == 960 && end == 960)
        return;

    if (begin == end) {
        s->fdsp->vector_fmul_scalar(output, output, gain_tab[end], FRAME_SAMPLES);
    } else {
        float a = gain_tab[begin] * (1.0f / (FRAME_SAMPLES - 1));
        float b = gain_tab[end  ] * (1.0f / (FRAME_SAMPLES - 1));
        int i;

        for (i = 0; i < FRAME_SAMPLES; i++)
            output[i] *= a * (FRAME_SAMPLES - i - 1) + b * i;
    }
}

static int filter_frame(DBEDecodeContext *s, AVFrame *frame)
{
    const DolbyEHeaderInfo *const metadata = &s->dectx.metadata;
    const uint8_t *reorder;
    int ch, ret;

    if (metadata->nb_channels == 4)
        reorder = ch_reorder_4;
    else if (metadata->nb_channels == 6)
        reorder = ch_reorder_6;
    else if (metadata->nb_programs == 1 && metadata->output_channel_order == CHANNEL_ORDER_DEFAULT)
        reorder = ch_reorder_8;
    else
        reorder = ch_reorder_n;

    frame->nb_samples = FRAME_SAMPLES;
    if ((ret = ff_get_buffer(s->avctx, frame, 0)) < 0)
        return ret;

    for (ch = 0; ch < metadata->nb_channels; ch++) {
        float *output = (float *)frame->extended_data[reorder[ch]];
        transform(s, &s->channels[0][ch], s->history[ch], output);
        transform(s, &s->channels[1][ch], s->history[ch], output + FRAME_SAMPLES / 2);
        apply_gain(s, metadata->begin_gain[ch], metadata->end_gain[ch], output);
    }

    return 0;
}

static int dolby_e_decode_frame(AVCodecContext *avctx, void *data,
                                int *got_frame_ptr, AVPacket *avpkt)
{
    DBEDecodeContext *s1 = avctx->priv_data;
    DBEContext *s = &s1->dectx;
    int i, j, ret;

    if ((ret = ff_dolby_e_parse_header(s, avpkt->data, avpkt->size)) < 0)
        return ret;

    if (s->metadata.nb_programs > 1 && !s->metadata.multi_prog_warned) {
        av_log(avctx, AV_LOG_WARNING, "Stream has %d programs (configuration %d), "
               "channels will be output in native order.\n",
               s->metadata.nb_programs, s->metadata.prog_conf);
        s->metadata.multi_prog_warned = 1;
    }

    av_channel_layout_uninit(&avctx->ch_layout);
    switch (s->metadata.nb_channels) {
    case 4:
        avctx->ch_layout = (AVChannelLayout)AV_CHANNEL_LAYOUT_4POINT0;
        break;
    case 6:
        avctx->ch_layout = (AVChannelLayout)AV_CHANNEL_LAYOUT_5POINT1;
        break;
    case 8:
        avctx->ch_layout = (AVChannelLayout)AV_CHANNEL_LAYOUT_7POINT1;
        break;
    default:
        avctx->ch_layout.order       = AV_CHANNEL_ORDER_UNSPEC;
        avctx->ch_layout.nb_channels = s->metadata.nb_channels;
        break;
    }

    avctx->sample_rate = s->metadata.sample_rate;
    avctx->sample_fmt  = AV_SAMPLE_FMT_FLTP;

    i = s->metadata.nb_channels / 2;
    j = s->metadata.nb_channels;
    if ((ret = parse_audio(s1, 0, i, 0)) < 0)
        return ret;
    if ((ret = parse_audio(s1, i, j, 0)) < 0)
        return ret;
    if ((ret = parse_metadata_ext(s1)) < 0)
        return ret;
    if ((ret = parse_audio(s1, 0, i, 1)) < 0)
        return ret;
    if ((ret = parse_audio(s1, i, j, 1)) < 0)
        return ret;
    if ((ret = parse_meter(s1)) < 0)
        return ret;
    if ((ret = filter_frame(s1, data)) < 0)
        return ret;

    *got_frame_ptr = 1;
    return avpkt->size;
}

static av_cold void dolby_e_flush(AVCodecContext *avctx)
{
    DBEDecodeContext *s = avctx->priv_data;

    memset(s->history, 0, sizeof(s->history));
}

static av_cold int dolby_e_close(AVCodecContext *avctx)
{
    DBEDecodeContext *s = avctx->priv_data;
    int i;

    for (i = 0; i < 3; i++)
        ff_mdct_end(&s->imdct[i]);

    av_freep(&s->fdsp);
    return 0;
}


static av_cold void init_tables(void)
{
    int i, j;

    for (i = 1; i < 17; i++)
        mantissa_tab1[i][0] = 1.0f / (1 << i - 1);

    for (i = 2; i < 16; i++) {
        mantissa_tab1[i][1] = 1.0f  / ((1 << i) - 1);
        mantissa_tab1[i][2] = 0.5f  / ((1 << i) - 1);
        mantissa_tab1[i][3] = 0.25f / ((1 << i) - 1);
    }

    mantissa_tab1[i][1] = 0.5f   / (1 << 15);
    mantissa_tab1[i][2] = 0.75f  / (1 << 15);
    mantissa_tab1[i][3] = 0.875f / (1 << 15);

    for (i = 1; i < 17; i++) {
        mantissa_tab2[i][1] = mantissa_tab1[i][0] * 0.5f;
        mantissa_tab2[i][2] = mantissa_tab1[i][0] * 0.75f;
        mantissa_tab2[i][3] = mantissa_tab1[i][0] * 0.875f;
        for (j = 1; j < 4; j++)
            mantissa_tab3[i][j] = 1.0f / (1 << i) + 1.0f / (1 << j) - 1.0f / (1 << i + j);
    }

    mantissa_tab3[1][3] = 0.6875f;

    for (i = 0; i < 25; i++) {
        exponent_tab[i * 2    ] = 1.0f      / (1 << i);
        exponent_tab[i * 2 + 1] = M_SQRT1_2 / (1 << i);
    }

    for (i = 1; i < 1024; i++)
        gain_tab[i] = exp2f((i - 960) / 64.0f);

    // short 1
    ff_kbd_window_init(window, 3.0f, 128);
    for (i = 0; i < 128; i++)
        window[128 + i] = window[127 - i];

    // start
    for (i = 0; i < 192; i++)
        window[256 + i] = start_window[i];

    // short 2
    for (i = 0; i < 192; i++)
        window[448 + i] = short_window2[i];
    for (i = 0; i < 64; i++)
        window[640 + i] = window[63 - i];

    // short 3
    for (i = 0; i < 64; i++)
        window[704 + i] = short_window3[i];
    for (i = 0; i < 192; i++)
        window[768 + i] = window[64 + i];

    // bridge
    for (i = 0; i < 128; i++)
        window[960 + i] = window[i];
    for (i = 0; i < 64; i++)
        window[1088 + i] = 1.0f;

    // long
    ff_kbd_window_init(window + 1408, 3.0f, 256);
    for (i = 0; i < 640; i++)
        window[1664 + i] = 1.0f;
    for (i = 0; i < 256; i++)
        window[2304 + i] = window[1152 + i] = window[1663 - i];

    // reverse start
    for (i = 0; i < 192; i++)
        window[2560 + i] = window[447 - i];

    // reverse short 2
    for (i = 0; i < 256; i++)
        window[2752 + i] = window[703 - i];

    // reverse short 3
    for (i = 0; i < 256; i++)
        window[3008 + i] = window[959 - i];

    // reverse bridge
    for (i = 0; i < 448; i++)
        window[3264 + i] = window[1407 - i];
}

static av_cold int dolby_e_init(AVCodecContext *avctx)
{
    static AVOnce init_once = AV_ONCE_INIT;
    DBEDecodeContext *s = avctx->priv_data;
    int i;

    if (ff_thread_once(&init_once, init_tables))
        return AVERROR_UNKNOWN;

    for (i = 0; i < 3; i++)
        if (ff_mdct_init(&s->imdct[i], imdct_bits_tab[i], 1, 2.0) < 0)
            return AVERROR(ENOMEM);

    if (!(s->fdsp = avpriv_float_dsp_alloc(0)))
        return AVERROR(ENOMEM);

#if FF_API_OLD_CHANNEL_LAYOUT
FF_DISABLE_DEPRECATION_WARNINGS
    if (avctx->request_channel_layout & AV_CH_LAYOUT_NATIVE)
        s->dectx.metadata.output_channel_order = CHANNEL_ORDER_CODED;
FF_ENABLE_DEPRECATION_WARNINGS
#endif

    s->dectx.metadata.multi_prog_warned = s->dectx.metadata.output_channel_order == CHANNEL_ORDER_CODED;
    s->dectx.avctx = s->avctx = avctx;
    return 0;
}

#define OFFSET(x) offsetof(DBEDecodeContext, x)
#define FLAGS (AV_OPT_FLAG_DECODING_PARAM | AV_OPT_FLAG_AUDIO_PARAM)
static const AVOption options[] = {
    { "channel_order", "Order in which the channels are to be exported",
        OFFSET(dectx.metadata.output_channel_order), AV_OPT_TYPE_INT,
        { .i64 = CHANNEL_ORDER_DEFAULT }, 0, 1, FLAGS, "channel_order" },
      { "default", "normal libavcodec channel order", 0, AV_OPT_TYPE_CONST,
        { .i64 = CHANNEL_ORDER_DEFAULT }, .flags = FLAGS, "channel_order" },
      { "coded",    "order in which the channels are coded in the bitstream",
        0, AV_OPT_TYPE_CONST, { .i64 = CHANNEL_ORDER_CODED }, .flags = FLAGS, "channel_order" },

      { NULL },
};

static const AVClass dolby_e_decoder_class = {
    .class_name = "Dolby E decoder",
    .item_name  = av_default_item_name,
    .option     = options,
    .version    = LIBAVUTIL_VERSION_INT,
};

const FFCodec ff_dolby_e_decoder = {
    .p.name         = "dolby_e",
    .p.long_name    = NULL_IF_CONFIG_SMALL("Dolby E"),
    .p.type         = AVMEDIA_TYPE_AUDIO,
    .p.id           = AV_CODEC_ID_DOLBY_E,
    .priv_data_size = sizeof(DBEDecodeContext),
    .p.priv_class   = &dolby_e_decoder_class,
    .init           = dolby_e_init,
    .decode         = dolby_e_decode_frame,
    .close          = dolby_e_close,
    .flush          = dolby_e_flush,
    .p.capabilities = AV_CODEC_CAP_DR1 | AV_CODEC_CAP_CHANNEL_CONF,
    .p.sample_fmts  = (const enum AVSampleFormat[]) { AV_SAMPLE_FMT_FLTP, AV_SAMPLE_FMT_NONE },
    .caps_internal  = FF_CODEC_CAP_INIT_THREADSAFE | FF_CODEC_CAP_INIT_CLEANUP,
};
